# Settings Page - API Documentation

This document outlines all API endpoints used in the settings page, their request/response formats, and behavior.

## Authentication

All endpoints require a valid JWT token in the `Authorization` header:

```
Authorization: Bearer <token>
```

---

## User Profile APIs

### 1. Get User Profile

**Endpoint:** `GET /users/profile`

**Response:**

```javascript
{
  success: true,
  data: {
    id: number,
    name: string,
    email: string,
    phone: string,
    created_at: string,
    updated_at: string
  }
}
```

**Behavior:**

- Fetches the current user's profile
- Updates `numus_user` in localStorage
- Populates the Account section form fields

---

### 2. Update User Profile

**Endpoint:** `PUT /users/profile`

**Request Body:**

```javascript
{
  name: string,
  email: string,
  phone: string
}
```

**Response:**

```javascript
{
  success: true,
  data: {
    id: number,
    name: string,
    email: string,
    phone: string,
    updated_at: string
  },
  message?: string
}
```

**Behavior:**

- Updates user profile information
- Returns updated user data
- Updates `numus_user` in localStorage

---

### 3. Change Password

**Endpoint:** `PUT /users/password`

**Request Body:**

```javascript
{
  current_password: string,
  new_password: string
}
```

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Validation:**

- Current password must be correct
- New password must be at least 8 characters
- Returns error if current password is incorrect

---

## Business APIs

### 4. Get Business Details

**Endpoint:** `GET /businesses/:id`

**Response:**

```javascript
{
  success: true,
  data: {
    id: number,
    name: string,
    email: string,
    phone: string,
    address: string,
    currency: string,
    default_delivery_price: number,
    description: string,
    subdomain: string,
    created_at: string,
    updated_at: string
  }
}
```

**Behavior:**

- Fetches complete business details
- Updates `numus_business` in localStorage
- Populates Business Profile form fields

---

### 5. Update Business

**Endpoint:** `PUT /businesses/:id`

**Request Body:**

```javascript
{
  name: string,
  email: string,
  phone: string,
  address: string,
  currency: string,
  default_delivery_price: number,
  description: string
}
```

**Response:**

```javascript
{
  success: true,
  data: {
    // Updated business object
  },
  message?: string
}
```

**Behavior:**

- Updates business information
- Merges updated data with existing business in localStorage

---

### 6. Update Subdomain

**Endpoint:** `PUT /businesses/:id/subdomain`

**Request Body:**

```javascript
{
  subdomain: string; // lowercase, alphanumeric and hyphens only, min 3 characters
}
```

**Response:**

```javascript
{
  success: true,
  message?: string  // e.g., "Subdomain updated successfully" or "Subdomain already taken"
}
```

**Validation:**

- Subdomain must be at least 3 characters
- Only lowercase letters, numbers, and hyphens allowed
- Must be unique across all businesses

---

## Settings APIs

### 7. Get Notification Settings

**Endpoint:** `GET /users/settings/notifications`

**Response:**

```javascript
{
  success: true,
  data: {
    newOrders: boolean,
    orderUpdates: boolean,
    driverUpdates: boolean,
    weeklyReports: boolean,
    marketing: boolean
  }
}
```

---

### 8. Update Notification Settings

**Endpoint:** `PUT /users/settings/notifications`

**Request Body:**

```javascript
{
  newOrders: boolean,
  orderUpdates: boolean,
  driverUpdates: boolean,
  weeklyReports: boolean,
  marketing: boolean
}
```

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Saves notification preferences
- Updates `numus_notification_settings` in localStorage

---

### 9. Get Preferences

**Endpoint:** `GET /users/settings/preferences`

**Response:**

```javascript
{
  success: true,
  data: {
    language: string,      // "en", "fr", "es", "pt"
    timezone: string,      // e.g., "Africa/Lagos"
    dateFormat: string     // "DD/MM/YYYY", "MM/DD/YYYY", "YYYY-MM-DD"
  }
}
```

---

### 10. Update Preferences

**Endpoint:** `PUT /users/settings/preferences`

**Request Body:**

```javascript
{
  language: string,
  timezone: string,
  dateFormat: string
}
```

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Saves user preferences
- Updates `numus_preferences` in localStorage

---

### 11. Get Order Settings

**Endpoint:** `GET /businesses/:businessId/settings/orders`

**Response:**

```javascript
{
  success: true,
  data: {
    autoAssign: boolean,      // Auto-assign drivers to new orders
    requireOtp: boolean,      // Require OTP for delivery completion
    sendSms: boolean          // Send SMS notifications to customers
  }
}
```

---

### 12. Update Order Settings

**Endpoint:** `PUT /businesses/:businessId/settings/orders`

**Request Body:**

```javascript
{
  autoAssign: boolean,
  requireOtp: boolean,
  sendSms: boolean
}
```

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Saves order settings for the business
- Updates `numus_order_settings` in localStorage

---

## Security & Session APIs

### 13. Logout All Sessions

**Endpoint:** `POST /auth/logout-all`

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Invalidates all active sessions except the current one
- User remains logged in on current device

---

## Danger Zone APIs

### 14. Export Business Data

**Endpoint:** `GET /businesses/:businessId/export`

**Response:**

```javascript
{
  success: true,
  data: {
    url: string  // Pre-signed URL for downloading the export file
  },
  message?: string
}
```

**Behavior:**

- Generates a complete export of business data
- Returns a download URL
- Opens the URL in a new tab for download

---

### 15. Delete All Orders

**Endpoint:** `DELETE /businesses/:businessId/orders/all`

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Permanently deletes all orders for the business
- Requires double confirmation from user
- Cannot be undone

---

### 16. Delete Account

**Endpoint:** `DELETE /users/account`

**Response:**

```javascript
{
  success: true,
  message?: string
}
```

**Behavior:**

- Schedules account for permanent deletion
- Requires typing "DELETE" to confirm
- Logs user out after 2 seconds
- Clears all localStorage data

---

## Error Handling

All endpoints follow the same error response format:

```javascript
{
  success: false,
  error: string | { message: string },
  message?: string
}
```

**HTTP Status Codes:**

- `401 Unauthorized` - Invalid or expired token, triggers logout
- `400 Bad Request` - Validation errors
- `403 Forbidden` - Insufficient permissions
- `404 Not Found` - Resource not found
- `500 Internal Server Error` - Server error

**Client Behavior:**

- All errors are caught and displayed as toast notifications
- Falls back to localStorage data if API fails (for GET requests)
- Logs errors to console for debugging
- 401 errors trigger automatic redirect to login page

---

## localStorage Keys

The following localStorage keys are updated by the settings page:

| Key                           | Content                  | Updated By            |
| ----------------------------- | ------------------------ | --------------------- |
| `numus_token`                 | JWT authentication token | Login                 |
| `numus_user`                  | User profile object      | Get/Update Profile    |
| `numus_business`              | Business details object  | Get/Update Business   |
| `numus_notification_settings` | Notification preferences | Update Notifications  |
| `numus_preferences`           | User preferences         | Update Preferences    |
| `numus_order_settings`        | Order settings           | Update Order Settings |

---

## Implementation Notes

1. **Data Fetching on Page Load:**

   - `loadBusinessData()` fetches from API and updates localStorage
   - `loadUserData()` fetches from API and updates localStorage
   - Falls back to localStorage if API fails

2. **Form Submissions:**

   - All forms use async/await for API calls
   - Show loading/success/error states via toast notifications
   - Update localStorage on successful updates
   - Display error messages from API responses

3. **Validation:**

   - Client-side validation before API calls (e.g., password length, subdomain format)
   - Server-side validation messages displayed to user
   - Form fields retain values on error

4. **Security:**
   - All API calls include JWT token
   - 401 responses trigger automatic logout
   - Sensitive operations require confirmation dialogs
   - Password changes require current password verification
