// Numus Logistics - API Service
class ApiService {
  constructor() {
    this.baseUrl = CONFIG.API_BASE_URL;
    this.token = localStorage.getItem('numus_token');
  }

  // Set authentication token
  setToken(token) {
    this.token = token;
    localStorage.setItem('numus_token', token);
  }

  // Clear authentication token
  clearToken() {
    this.token = null;
    localStorage.removeItem('numus_token');
  }

  // Get headers for requests
  getHeaders(includeAuth = true) {
    const headers = {
      'Content-Type': 'application/json',
    };
    if (includeAuth && this.token) {
      headers['Authorization'] = `Bearer ${this.token}`;
    }
    return headers;
  }

  // Generic request method
  async request(endpoint, options = {}) {
    const url = `${this.baseUrl}${endpoint}`;
    const config = {
      ...options,
      headers: {
        ...this.getHeaders(options.auth !== false),
        ...options.headers,
      },
    };

    try {
      const response = await fetch(url, config);
      const data = await response.json();

      if (!response.ok) {
        if (response.status === 401) {
          this.clearToken();
          window.location.href = 'login';
          throw new Error('Session expired. Please login again.');
        }
        // Check multiple possible error message locations
        const errorMessage = data.error?.message || data.message || data.error || 'Request failed';
        throw new Error(errorMessage);
      }

      return data;
    } catch (error) {
      console.error('API Error:', error);
      throw error;
    }
  }

  // GET request
  async get(endpoint, params = {}) {
    // Filter out undefined, null, and empty string values
    const cleanParams = Object.entries(params).reduce((acc, [key, value]) => {
      if (value !== undefined && value !== null && value !== '') {
        acc[key] = value;
      }
      return acc;
    }, {});

    const queryString = new URLSearchParams(cleanParams).toString();
    const url = queryString ? `${endpoint}?${queryString}` : endpoint;
    return this.request(url, { method: 'GET' });
  }

  // POST request
  async post(endpoint, data = {}, options = {}) {
    return this.request(endpoint, {
      method: 'POST',
      body: JSON.stringify(data),
      ...options,
    });
  }

  // PUT request
  async put(endpoint, data = {}) {
    return this.request(endpoint, {
      method: 'PUT',
      body: JSON.stringify(data),
    });
  }

  // DELETE request
  async delete(endpoint) {
    return this.request(endpoint, { method: 'DELETE' });
  }

  // ==================== Authentication ====================
  
  async login(email, password) {
    const response = await this.post('/auth/login', { email, password }, { auth: false });
    if (response.success && response.data.token) {
      this.setToken(response.data.token);
      localStorage.setItem('numus_user', JSON.stringify(response.data.user));
      if (response.data.business) {
        localStorage.setItem('numus_business', JSON.stringify(response.data.business));
      }
    }
    return response;
  }

  async register(userData) {
    const response = await this.post('/auth/register', userData, { auth: false });
    if (response.success && response.data.token) {
      this.setToken(response.data.token);
      localStorage.setItem('numus_user', JSON.stringify(response.data.user));
    }
    return response;
  }

  async getCurrentUser() {
    return this.get('/auth/me');
  }

  logout() {
    this.clearToken();
    localStorage.removeItem('numus_user');
    localStorage.removeItem('numus_business');
    window.location.href = 'login';
  }

  isAuthenticated() {
    return !!this.token;
  }

  // ==================== Business ====================

  async createBusiness(businessData) {
    const response = await this.post('/businesses', businessData);
    if (response.success) {
      localStorage.setItem('numus_business', JSON.stringify(response.data));
    }
    return response;
  }

  async getBusiness(id) {
    return this.get(`/businesses/${id}`);
  }

  async updateBusiness(id, data) {
    return this.put(`/businesses/${id}`, data);
  }

  async getBusinessStats(businessId) {
    return this.get(`/businesses/${businessId}/stats`);
  }

  // ==================== Drivers ====================

  async getDrivers(params = {}) {
    const businessId = this.getBusinessId();
    return this.get('/drivers', { business_id: businessId, ...params });
  }

  async getDriver(id) {
    if (!id || id === 'undefined' || id === 'null') {
      throw new Error('Invalid driver ID');
    }
    console.log('API getDriver called with ID:', id);
    return this.get(`/drivers/${id}`);
  }

  async createDriver(driverData) {
    const businessId = this.getBusinessId();
    return this.post('/drivers', { business_id: businessId, ...driverData });
  }

  async updateDriver(id, data) {
    if (!id || id === 'undefined' || id === 'null') {
      throw new Error('Invalid driver ID');
    }
    console.log('API updateDriver called with ID:', id);
    return this.put(`/drivers/${id}`, data);
  }

  async deleteDriver(id) {
    if (!id || id === 'undefined' || id === 'null') {
      throw new Error('Invalid driver ID');
    }
    console.log('API deleteDriver called with ID:', id);
    return this.delete(`/drivers/${id}`);
  }

  async updateDriverStatus(id, status) {
    if (!id || id === 'undefined' || id === 'null') {
      throw new Error('Invalid driver ID');
    }
    console.log('API updateDriverStatus called with ID:', id, 'Status:', status);
    return this.put(`/drivers/${id}/status`, { status });
  }

  async getDriverOrders(driverId, params = {}) {
    if (!driverId || driverId === 'undefined' || driverId === 'null') {
      throw new Error('Invalid driver ID');
    }
    console.log('API getDriverOrders called with ID:', driverId);
    return this.get(`/drivers/${driverId}/orders`, params);
  }

  // ==================== Orders ====================

  async getOrders(params = {}) {
    const businessId = this.getBusinessId();
    return this.get('/orders', { business_id: businessId, ...params });
  }

  async getOrder(id) {
    return this.get(`/orders/${id}`);
  }

  async createOrder(orderData) {
    const businessId = this.getBusinessId();
    return this.post('/orders', { business_id: businessId, ...orderData });
  }

  async updateOrder(id, data) {
    return this.put(`/orders/${id}`, data);
  }

  async cancelOrder(id) {
    return this.delete(`/orders/${id}`);
  }

  async assignDriver(orderId, driverId) {
    return this.post(`/orders/${orderId}/assign`, { driver_id: driverId });
  }

  async updateOrderStatus(orderId, status, location = null) {
    const data = { status };
    if (location) {
      data.location_lat = location.lat;
      data.location_lng = location.lng;
    }
    return this.put(`/orders/${orderId}/status`, data);
  }

  async trackOrder(orderId) {
    return this.get(`/orders/track/${orderId}`);
  }

  // ==================== OTPs ====================

  async generateOTP(orderId, otpType) {
    return this.post(`/orders/${orderId}/otp/generate`, { otp_type: otpType });
  }

  async verifyOTP(orderId, otpType, otpCode) {
    return this.post(`/orders/${orderId}/otp/verify`, {
      otp_type: otpType,
      otp_code: otpCode,
    });
  }

  // ==================== Pricing ====================

  async getPricing(params = {}) {
    const businessId = this.getBusinessId();
    return this.get('/pricing', { business_id: businessId, ...params });
  }

  async getPricingRoute(id) {
    return this.get(`/pricing/${id}`);
  }

  async createPricing(pricingData) {
    const businessId = this.getBusinessId();
    return this.post('/pricing', { business_id: businessId, ...pricingData });
  }

  async updatePricing(id, data) {
    return this.put(`/pricing/${id}`, data);
  }

  async deletePricing(id) {
    return this.delete(`/pricing/${id}`);
  }

  async calculatePrice(routeId, distanceKm, weightKg) {
    const businessId = this.getBusinessId();
    return this.post('/pricing/calculate', {
      business_id: businessId,
      route_id: routeId,
      distance_km: distanceKm,
      weight_kg: weightKg,
    });
  }

  // ==================== Reports ====================

  async getOrdersReport(params) {
    const businessId = this.getBusinessId();
    return this.get('/reports/orders', { business_id: businessId, ...params });
  }

  async getDriversReport(params) {
    const businessId = this.getBusinessId();
    return this.get('/reports/drivers', { business_id: businessId, ...params });
  }

  // ==================== Live Tracking ====================

  async getDriverLocations() {
    const businessId = this.getBusinessId();
    return this.get('/drivers/locations', { business_id: businessId });
  }

  // ==================== Website Builder ====================

  async getWebsiteConfig() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/website`);
  }

  async saveWebsiteConfig(config) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/website`, { website_config: config });
  }

  async publishWebsite(config) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/website/publish`, { website_config: config });
  }

  // ==================== Domains ====================

  async getCustomDomains() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/domains`);
  }

  async addCustomDomain(domain) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/domains`, { domain });
  }

  async verifyDomain(domainId) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/domains/${domainId}/verify`);
  }

  async setPrimaryDomain(domainId) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/domains/${domainId}/primary`);
  }

  async removeDomain(domainId) {
    const businessId = this.getBusinessId();
    return this.delete(`/businesses/${businessId}/domains/${domainId}`);
  }

  async updateSubdomain(subdomain) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/subdomain`, { subdomain });
  }

  // ==================== Payment Settings ====================

  async getPaymentSettings() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/payments`);
  }

  async updatePaymentSettings(settings) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/payments`, settings);
  }

  async testPaymentGateway(gatewayId, keys) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/payments/test`, { gateway: gatewayId, keys });
  }

  // ==================== Subscription ====================

  async getSubscription() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/subscription`);
  }

  async verifySubscriptionPayment(paymentData) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/subscription/verify`, paymentData);
  }

  async cancelSubscription() {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/subscription/cancel`);
  }

  async getSubscriptionUsage() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/subscription/usage`);
  }

  // ==================== WhatsApp Integration ====================

  async getWhatsAppConfig() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/whatsapp`);
  }

  async connectWhatsApp(config) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/whatsapp/connect`, config);
  }

  async disconnectWhatsApp() {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/whatsapp/disconnect`);
  }

  async testWhatsAppConnection() {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/whatsapp/test`);
  }

  // ==================== User Profile ====================

  async getUserProfile() {
    return this.get('/users/profile');
  }

  async updateUserProfile(data) {
    return this.put('/users/profile', data);
  }

  async changePassword(currentPassword, newPassword) {
    return this.put('/users/password', {
      current_password: currentPassword,
      new_password: newPassword
    });
  }

  // ==================== Team Management ====================

  async getTeamMembers() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/team`);
  }

  async inviteTeamMember(data) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/team/invite`, data);
  }

  async updateTeamMember(memberId, data) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/team/${memberId}`, data);
  }

  async removeTeamMember(memberId) {
    const businessId = this.getBusinessId();
    return this.delete(`/businesses/${businessId}/team/${memberId}`);
  }

  async resendTeamInvite(memberId) {
    const businessId = this.getBusinessId();
    return this.post(`/businesses/${businessId}/team/${memberId}/resend`);
  }

  async acceptTeamInvite(token) {
    return this.post('/team/accept-invite', { token }, { auth: false });
  }

  // ==================== Settings ====================

  async getNotificationSettings() {
    return this.get('/users/settings/notifications');
  }

  async updateNotificationSettings(settings) {
    return this.put('/users/settings/notifications', settings);
  }

  async getPreferences() {
    return this.get('/users/settings/preferences');
  }

  async updatePreferences(preferences) {
    return this.put('/users/settings/preferences', preferences);
  }

  async getOrderSettings() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/settings/orders`);
  }

  async updateOrderSettings(settings) {
    const businessId = this.getBusinessId();
    return this.put(`/businesses/${businessId}/settings/orders`, settings);
  }

  async logoutAllSessions() {
    return this.post('/auth/logout-all');
  }

  async exportBusinessData() {
    const businessId = this.getBusinessId();
    return this.get(`/businesses/${businessId}/export`);
  }

  async deleteAllOrders() {
    const businessId = this.getBusinessId();
    return this.delete(`/businesses/${businessId}/orders/all`);
  }

  async deleteAccount() {
    return this.delete('/users/account');
  }

  // ==================== Utility Methods ====================

  getBusinessId() {
    const business = JSON.parse(localStorage.getItem('numus_business') || '{}');
    return business.id;
  }

  getCurrentUserId() {
    const user = JSON.parse(localStorage.getItem('numus_user') || '{}');
    return user.id;
  }
}

// Create singleton instance
const api = new ApiService();
