// Numus Logistics - Geocoding and Distance Calculation

/**
 * Geocode an address to get latitude and longitude
 * Uses Google Maps Geocoding API if available, otherwise falls back to Nominatim (OpenStreetMap)
 */
async function geocodeAddress(address) {
  if (!address || address.trim() === '') {
    return null;
  }

  // Check if Google Maps API key is configured
  if (CONFIG.GOOGLE_MAPS_API_KEY && CONFIG.GOOGLE_MAPS_API_KEY !== 'YOUR_GOOGLE_MAPS_API_KEY') {
    const result = await geocodeWithGoogle(address);
    if (result && validateCoordinates(result.lat, result.lng, 'Nigeria')) {
      return result;
    }
  }

  // Fallback to free Nominatim API (OpenStreetMap)
  const result = await geocodeWithNominatim(address);
  if (result && validateCoordinates(result.lat, result.lng, 'Nigeria')) {
    return result;
  }

  return null;
}

/**
 * Validate that coordinates are within expected range for a country
 * Nigeria approximate bounds: Lat 4° to 14°N, Lng 2.5° to 15°E
 */
function validateCoordinates(lat, lng, country = 'Nigeria') {
  if (country === 'Nigeria') {
    // Nigeria bounds with some margin
    const isValid = lat >= 4 && lat <= 14 && lng >= 2.5 && lng <= 15;
    if (!isValid) {
      console.warn(`Coordinates (${lat}, ${lng}) are outside Nigeria bounds`);
    }
    return isValid;
  }
  return true;
}

/**
 * Geocode using Google Maps Geocoding API
 */
async function geocodeWithGoogle(address) {
  try {
    const encodedAddress = encodeURIComponent(address);
    const url = `https://maps.googleapis.com/maps/api/geocode/json?address=${encodedAddress}&key=${CONFIG.GOOGLE_MAPS_API_KEY}`;

    const response = await fetch(url);
    const data = await response.json();

    if (data.status === 'OK' && data.results.length > 0) {
      const location = data.results[0].geometry.location;
      return {
        lat: location.lat,
        lng: location.lng,
        formatted_address: data.results[0].formatted_address
      };
    } else {
      console.error('Google Geocoding error:', data.status);
      return null;
    }
  } catch (error) {
    console.error('Error geocoding with Google:', error);
    return null;
  }
}

/**
 * Geocode using Nominatim (OpenStreetMap) - Free alternative
 */
async function geocodeWithNominatim(address) {
  try {
    // Add Nigeria and Lagos context to improve accuracy for local addresses
    let addressWithContext = address;

    // If address doesn't mention Nigeria or Lagos, add context
    if (!address.toLowerCase().includes('nigeria') && !address.toLowerCase().includes('lagos')) {
      addressWithContext = `${address}, Lagos, Nigeria`;
    } else if (!address.toLowerCase().includes('nigeria')) {
      addressWithContext = `${address}, Nigeria`;
    }

    const encodedAddress = encodeURIComponent(addressWithContext);

    // Nominatim with Nigeria country code for better accuracy
    const url = `https://nominatim.openstreetmap.org/search?q=${encodedAddress}&format=json&limit=1&countrycodes=ng&addressdetails=1`;

    const response = await fetch(url, {
      headers: {
        'User-Agent': 'NumusLogistics/1.0'
      }
    });

    // Nominatim requires a delay between requests (max 1 request per second)
    await new Promise(resolve => setTimeout(resolve, 1000));

    const data = await response.json();

    if (data.length > 0) {
      const result = {
        lat: parseFloat(data[0].lat),
        lng: parseFloat(data[0].lon),
        formatted_address: data[0].display_name
      };
      console.log('Geocoded:', address, '→', result);
      return result;
    } else {
      console.error('Nominatim: No results found for', address);
      return null;
    }
  } catch (error) {
    console.error('Error geocoding with Nominatim:', error);
    return null;
  }
}

/**
 * Calculate distance between two coordinates using Haversine formula
 * Returns distance in kilometers
 */
function calculateDistance(lat1, lng1, lat2, lng2) {
  const R = 6371; // Radius of the Earth in kilometers
  const dLat = toRadians(lat2 - lat1);
  const dLng = toRadians(lng2 - lng1);

  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.cos(toRadians(lat1)) * Math.cos(toRadians(lat2)) *
    Math.sin(dLng / 2) * Math.sin(dLng / 2);

  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  const distance = R * c;

  return distance;
}

/**
 * Convert degrees to radians
 */
function toRadians(degrees) {
  return degrees * (Math.PI / 180);
}

/**
 * Geocode addresses and calculate distance
 * This function is called when user finishes entering pickup or delivery address
 */
async function geocodeAndCalculateDistance() {
  const pickupAddress = document.getElementById('pickupAddress').value.trim();
  const deliveryAddress = document.getElementById('deliveryAddress').value.trim();

  // Only proceed if both addresses are filled
  if (!pickupAddress || !deliveryAddress) {
    return;
  }

  // Show loading state
  const distanceInput = document.getElementById('distanceInput');
  const originalValue = distanceInput.value;
  distanceInput.value = '';
  distanceInput.placeholder = 'Calculating...';

  try {
    // Geocode both addresses
    const [pickupResult, deliveryResult] = await Promise.all([
      geocodeAddress(pickupAddress),
      geocodeAddress(deliveryAddress)
    ]);

    if (!pickupResult) {
      Utils.showToast('Could not find pickup address in Nigeria. Please use a more specific address (e.g., include street name and area).', 'warning');
      distanceInput.placeholder = '0.0';
      distanceInput.value = originalValue;
      return;
    }

    if (!deliveryResult) {
      Utils.showToast('Could not find delivery address in Nigeria. Please use a more specific address (e.g., include street name and area).', 'warning');
      distanceInput.placeholder = '0.0';
      distanceInput.value = originalValue;
      return;
    }

    // Store coordinates in hidden fields
    document.getElementById('pickupLat').value = pickupResult.lat;
    document.getElementById('pickupLng').value = pickupResult.lng;
    document.getElementById('deliveryLat').value = deliveryResult.lat;
    document.getElementById('deliveryLng').value = deliveryResult.lng;

    // Debug log
    console.log('Pickup coords:', pickupResult.lat, pickupResult.lng);
    console.log('Delivery coords:', deliveryResult.lat, deliveryResult.lng);

    // Calculate distance
    const distance = calculateDistance(
      pickupResult.lat,
      pickupResult.lng,
      deliveryResult.lat,
      deliveryResult.lng
    );

    console.log('Calculated distance:', distance, 'km');

    // Update distance field (rounded to 1 decimal place)
    const roundedDistance = Math.round(distance * 10) / 10;
    distanceInput.value = roundedDistance;
    distanceInput.placeholder = '0.0';

    // Trigger price calculation
    calculateOrderPrice();

    // Show success message with coordinate info for debugging
    Utils.showToast(`Distance calculated: ${roundedDistance} km`, 'success');

  } catch (error) {
    console.error('Error in geocodeAndCalculateDistance:', error);
    Utils.showToast('Error calculating distance. Please try again.', 'error');
    distanceInput.placeholder = '0.0';
    distanceInput.value = originalValue;
  }
}
