// Numus Logistics - Subscription Limit Checker
// Include this before main page scripts to enforce plan limits

const PLAN_LIMITS = {
  starter: {
    orders: 50,
    drivers: 3,
    whatsappMessages: 100,
    features: {
      customDomain: false,
      paymentGateway: false,
      codCollection: false,
      whiteLabel: false
    }
  },
  growth: {
    orders: -1, // unlimited
    drivers: -1,
    whatsappMessages: -1,
    features: {
      customDomain: true,
      paymentGateway: true,
      codCollection: true,
      whiteLabel: true
    }
  }
};

class SubscriptionChecker {
  constructor() {
    this.plan = 'starter';
    this.usage = { orders: 0, drivers: 0, whatsappMessages: 0 };
    this.loaded = false;
  }

  async init() {
    if (this.loaded) return;
    
    try {
      const response = await api.getSubscription();
      if (response.success && response.data) {
        this.plan = response.data.plan || 'starter';
        this.usage = response.data.usage || this.usage;
      }
    } catch (error) {
      console.log('Using default subscription');
    }
    
    this.loaded = true;
    this.showPlanBadge();
  }

  showPlanBadge() {
    // Add plan badge to header if on Growth plan
    if (this.plan === 'growth') {
      const header = document.querySelector('.main-header .header-right');
      if (header && !document.getElementById('planBadge')) {
        const badge = document.createElement('span');
        badge.id = 'planBadge';
        badge.style.cssText = 'background: linear-gradient(135deg, #f97316, #ea580c); color: white; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; margin-right: 12px;';
        badge.innerHTML = '🔥 Growth';
        header.insertBefore(badge, header.firstChild);
      }
    }
  }

  getLimits() {
    return PLAN_LIMITS[this.plan] || PLAN_LIMITS.starter;
  }

  canCreateOrder() {
    const limits = this.getLimits();
    if (limits.orders === -1) return { allowed: true };
    return {
      allowed: this.usage.orders < limits.orders,
      remaining: limits.orders - this.usage.orders,
      limit: limits.orders
    };
  }

  canAddDriver() {
    const limits = this.getLimits();
    if (limits.drivers === -1) return { allowed: true };
    return {
      allowed: this.usage.drivers < limits.drivers,
      remaining: limits.drivers - this.usage.drivers,
      limit: limits.drivers
    };
  }

  hasFeature(feature) {
    const limits = this.getLimits();
    return limits.features[feature] || false;
  }

  isGrowth() {
    return this.plan === 'growth';
  }

  showUpgradeModal(message) {
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.id = 'upgradeRequiredModal';
    modal.style.display = 'flex';
    modal.innerHTML = `
      <div class="modal" style="max-width: 420px;">
        <div class="modal-header">
          <h3>Upgrade Required</h3>
          <button class="modal-close" onclick="document.getElementById('upgradeRequiredModal').remove()">×</button>
        </div>
        <div class="modal-body" style="text-align: center; padding: 32px;">
          <div style="font-size: 64px; margin-bottom: 16px;">🔒</div>
          <h3 style="margin-bottom: 8px;">Plan Limit Reached</h3>
          <p style="color: var(--color-text-secondary); margin-bottom: 24px;">${message}</p>
          <a href="subscription" class="btn btn-accent" style="width: 100%;">
            🔥 Upgrade to Growth
          </a>
          <p style="font-size: 12px; color: var(--color-text-secondary); margin-top: 16px;">
            Get unlimited orders, drivers, and premium features
          </p>
        </div>
      </div>
    `;
    document.body.appendChild(modal);
  }

  showFeatureLockedModal(featureName) {
    this.showUpgradeModal(`${featureName} is only available on the Growth plan. Upgrade now to unlock this feature.`);
  }
}

// Global instance
const subscriptionChecker = new SubscriptionChecker();

// Auto-initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  if (typeof api !== 'undefined') {
    subscriptionChecker.init();
  }
});

// Helper functions for easy access
function checkOrderLimit() {
  const check = subscriptionChecker.canCreateOrder();
  if (!check.allowed) {
    subscriptionChecker.showUpgradeModal(
      `You've reached your monthly order limit (${check.limit}). Upgrade to Growth for unlimited orders.`
    );
    return false;
  }
  return true;
}

function checkDriverLimit() {
  const check = subscriptionChecker.canAddDriver();
  if (!check.allowed) {
    subscriptionChecker.showUpgradeModal(
      `You've reached your driver limit (${check.limit}). Upgrade to Growth for unlimited drivers.`
    );
    return false;
  }
  return true;
}

function checkFeature(feature, featureName) {
  if (!subscriptionChecker.hasFeature(feature)) {
    subscriptionChecker.showFeatureLockedModal(featureName);
    return false;
  }
  return true;
}
