// Numus Logistics - Subscription Management
// Global pricing with local currency support

// =============================================================================
// PRICING STRUCTURE & COST ANALYSIS
// =============================================================================
// 
// OPERATIONAL COSTS (per user/month at scale):
// - Xano API calls: ~$0.50-1.00
// - WhatsApp (Twilio): ~$0.50-2.00
// - Server hosting: ~$0.30
// - Payment processing: 1.5-3%
// - Support overhead: ~$1.00
// TOTAL COST: ~$3-5 per active user
//
// PRICING STRATEGY:
// - Starter (Free): Lead generation, limited features
// - Growth ($29/mo): Main revenue driver, all features unlimited
// - Annual discount: 20% (improves cash flow, reduces churn)
//
// BREAK-EVEN: ~100 paying users
// PROFITABILITY TARGET: 500+ Growth users = $14,500/mo gross
// =============================================================================

const SUBSCRIPTION_CONFIG = {
  // Paystack public keys (replace with your actual keys)
  PAYSTACK_PUBLIC_KEY: 'pk_live_4539d66b6c0c5df17d99ea0f010bb4baf15dd7a5',
  
  // Plan definitions
  PLANS: {
    starter: {
      id: 'starter',
      name: 'Starter',
      icon: '🚀',
      tagline: 'Perfect for getting started',
      monthlyPrice: 0,
      annualPrice: 0,
      limits: {
        orders: 50,
        drivers: 3,
        whatsappMessages: 100,
        customDomain: false,
        paymentGateway: false,
        codCollection: false,
        prioritySupport: false,
        whiteLabel: false,
        apiAccess: false,
      },
      features: [
        { name: 'Orders', limit: '50/month', included: true },
        { name: 'Drivers', limit: '3 max', included: true },
        { name: 'WhatsApp Messages', limit: '100/month', included: true },
        { name: 'Website Builder', included: true },
        { name: 'Numus Subdomain', included: true },
        { name: 'Live GPS Tracking', included: true },
        { name: 'OTP Verification', included: true },
        { name: 'Custom Domain', included: false },
        { name: 'Payment Gateway', included: false },
        { name: 'COD Collection', included: false },
        { name: 'Priority Support', included: false },
        { name: 'White-label', included: false },
      ]
    },
    growth: {
      id: 'growth',
      name: 'Growth',
      icon: '🔥',
      tagline: 'Scale your logistics business',
      monthlyPrice: 29, // USD base price
      annualPrice: 278, // 20% discount (29 * 12 * 0.8)
      limits: {
        orders: -1, // -1 = unlimited
        drivers: -1,
        whatsappMessages: -1,
        customDomain: true,
        paymentGateway: true,
        codCollection: true,
        prioritySupport: true,
        whiteLabel: true,
        apiAccess: true,
      },
      features: [
        { name: 'Orders', limit: 'Unlimited', included: true },
        { name: 'Drivers', limit: 'Unlimited', included: true },
        { name: 'WhatsApp Messages', limit: 'Unlimited', included: true },
        { name: 'Website Builder', included: true },
        { name: 'Numus Subdomain', included: true },
        { name: 'Live GPS Tracking', included: true },
        { name: 'OTP Verification', included: true },
        { name: 'Custom Domain', included: true },
        { name: 'Payment Gateway Integration', included: true },
        { name: 'COD Collection & Management', included: true },
        { name: 'Priority Support (24/7)', included: true },
        { name: 'White-label (Remove Numus branding)', included: true },
      ]
    }
  },

  // Currency configurations by country
  // Prices are calibrated based on purchasing power parity (PPP)
  CURRENCIES: {
    // Africa
    NG: { code: 'NGN', symbol: '₦', name: 'Nigerian Naira', rate: 1600, monthlyPrice: 25000, flag: '🇳🇬' },
    GH: { code: 'GHS', symbol: '₵', name: 'Ghanaian Cedi', rate: 15, monthlyPrice: 350, flag: '🇬🇭' },
    KE: { code: 'KES', symbol: 'KSh', name: 'Kenyan Shilling', rate: 155, monthlyPrice: 3500, flag: '🇰🇪' },
    ZA: { code: 'ZAR', symbol: 'R', name: 'South African Rand', rate: 19, monthlyPrice: 450, flag: '🇿🇦' },
    EG: { code: 'EGP', symbol: 'E£', name: 'Egyptian Pound', rate: 50, monthlyPrice: 1200, flag: '🇪🇬' },
    
    // Americas
    US: { code: 'USD', symbol: '$', name: 'US Dollar', rate: 1, monthlyPrice: 29, flag: '🇺🇸' },
    CA: { code: 'CAD', symbol: 'C$', name: 'Canadian Dollar', rate: 1.4, monthlyPrice: 39, flag: '🇨🇦' },
    BR: { code: 'BRL', symbol: 'R$', name: 'Brazilian Real', rate: 5, monthlyPrice: 120, flag: '🇧🇷' },
    MX: { code: 'MXN', symbol: 'MX$', name: 'Mexican Peso', rate: 18, monthlyPrice: 450, flag: '🇲🇽' },
    
    // Europe
    GB: { code: 'GBP', symbol: '£', name: 'British Pound', rate: 0.8, monthlyPrice: 24, flag: '🇬🇧' },
    EU: { code: 'EUR', symbol: '€', name: 'Euro', rate: 0.95, monthlyPrice: 27, flag: '🇪🇺' },
    DE: { code: 'EUR', symbol: '€', name: 'Euro', rate: 0.95, monthlyPrice: 27, flag: '🇩🇪' },
    FR: { code: 'EUR', symbol: '€', name: 'Euro', rate: 0.95, monthlyPrice: 27, flag: '🇫🇷' },
    
    // Asia
    IN: { code: 'INR', symbol: '₹', name: 'Indian Rupee', rate: 83, monthlyPrice: 1999, flag: '🇮🇳' },
    PK: { code: 'PKR', symbol: '₨', name: 'Pakistani Rupee', rate: 280, monthlyPrice: 6000, flag: '🇵🇰' },
    PH: { code: 'PHP', symbol: '₱', name: 'Philippine Peso', rate: 56, monthlyPrice: 1400, flag: '🇵🇭' },
    ID: { code: 'IDR', symbol: 'Rp', name: 'Indonesian Rupiah', rate: 16000, monthlyPrice: 400000, flag: '🇮🇩' },
    MY: { code: 'MYR', symbol: 'RM', name: 'Malaysian Ringgit', rate: 4.7, monthlyPrice: 120, flag: '🇲🇾' },
    SG: { code: 'SGD', symbol: 'S$', name: 'Singapore Dollar', rate: 1.35, monthlyPrice: 39, flag: '🇸🇬' },
    AE: { code: 'AED', symbol: 'د.إ', name: 'UAE Dirham', rate: 3.67, monthlyPrice: 99, flag: '🇦🇪' },
    SA: { code: 'SAR', symbol: '﷼', name: 'Saudi Riyal', rate: 3.75, monthlyPrice: 99, flag: '🇸🇦' },
    
    // Oceania
    AU: { code: 'AUD', symbol: 'A$', name: 'Australian Dollar', rate: 1.55, monthlyPrice: 45, flag: '🇦🇺' },
    NZ: { code: 'NZD', symbol: 'NZ$', name: 'New Zealand Dollar', rate: 1.7, monthlyPrice: 49, flag: '🇳🇿' },
    
    // Default fallback
    DEFAULT: { code: 'USD', symbol: '$', name: 'US Dollar', rate: 1, monthlyPrice: 29, flag: '🌍' }
  }
};

// State
let subscriptionState = {
  currentPlan: 'starter',
  isAnnual: false,
  userCountry: 'US',
  currency: SUBSCRIPTION_CONFIG.CURRENCIES.DEFAULT,
  usage: {
    orders: 0,
    drivers: 0,
    whatsappMessages: 0
  },
  subscription: null,
  billingHistory: []
};

// =============================================================================
// INITIALIZATION
// =============================================================================

async function initSubscriptionPage() {
  await detectUserLocation();
  await loadSubscriptionData();
  updateUI();

  // Add event listener for growth button as backup
  const growthBtn = document.getElementById('growthBtn');
  if (growthBtn) {
    growthBtn.addEventListener('click', function(e) {
      e.preventDefault();
      e.stopPropagation();
      if (!this.disabled) {
        selectPlan('growth');
      }
    });
  }
}

async function detectUserLocation() {
  try {
    // Try to get location from IP
    const response = await fetch('https://ipapi.co/json/');
    const data = await response.json();
    const countryCode = data.country_code;
    
    if (SUBSCRIPTION_CONFIG.CURRENCIES[countryCode]) {
      subscriptionState.userCountry = countryCode;
      subscriptionState.currency = SUBSCRIPTION_CONFIG.CURRENCIES[countryCode];
    } else {
      // Default to USD for unsupported countries
      subscriptionState.currency = SUBSCRIPTION_CONFIG.CURRENCIES.DEFAULT;
    }
  } catch (error) {
    console.log('Could not detect location, using default currency');
    subscriptionState.currency = SUBSCRIPTION_CONFIG.CURRENCIES.DEFAULT;
  }
}

async function loadSubscriptionData() {
  try {
    const response = await api.getSubscription();
    if (response.success && response.data) {
      subscriptionState.currentPlan = response.data.plan || 'starter';
      subscriptionState.subscription = response.data;
      subscriptionState.usage = response.data.usage || subscriptionState.usage;
      subscriptionState.billingHistory = response.data.billing_history || [];
    }
  } catch (error) {
    console.log('Using default subscription state');
  }
}

// =============================================================================
// UI UPDATES
// =============================================================================

function updateUI() {
  updatePlanCards();
  updateUsageStats();
  updateBillingHistory();
  checkLimits();
}

function updatePlanCards() {
  const currency = subscriptionState.currency;
  const isAnnual = subscriptionState.isAnnual;
  const currentPlan = subscriptionState.currentPlan;

  // Update Growth plan pricing
  const monthlyPrice = currency.monthlyPrice;
  const annualPrice = Math.round(monthlyPrice * 12 * 0.8); // 20% discount
  const displayPrice = isAnnual ? Math.round(annualPrice / 12) : monthlyPrice;

  document.getElementById('growthCurrency').textContent = currency.symbol;
  document.getElementById('growthAmount').textContent = formatNumber(displayPrice);
  document.getElementById('growthPeriod').textContent = isAnnual ? '/month (billed annually)' : '/month';
  
  if (isAnnual) {
    document.getElementById('growthOriginal').textContent = currency.symbol + formatNumber(monthlyPrice);
  } else {
    document.getElementById('growthOriginal').textContent = '';
  }

  // Update Starter currency symbol
  document.getElementById('starterCurrency').textContent = currency.symbol;

  // Update current plan indicators
  const starterCard = document.getElementById('starterCard');
  const growthCard = document.getElementById('growthCard');
  const starterBtn = document.getElementById('starterBtn');
  const growthBtn = document.getElementById('growthBtn');

  starterCard.classList.toggle('current', currentPlan === 'starter');
  growthCard.classList.toggle('current', currentPlan === 'growth');

  if (currentPlan === 'starter') {
    starterBtn.textContent = 'Current Plan';
    starterBtn.disabled = true;
    starterBtn.classList.add('btn-secondary');
    starterBtn.classList.remove('btn-primary');
    
    growthBtn.textContent = 'Upgrade to Growth';
    growthBtn.disabled = false;
  } else {
    starterBtn.textContent = 'Downgrade';
    starterBtn.disabled = false;
    starterBtn.classList.remove('btn-secondary');
    starterBtn.classList.add('btn-outline');
    
    growthBtn.textContent = 'Current Plan';
    growthBtn.disabled = true;
  }

  // Update header
  document.getElementById('currentPlanName').textContent =
    currentPlan === 'growth' ? 'Growth' : 'Starter (Free)';

  // Update billing period display
  const billingPeriodEl = document.getElementById('billingPeriod');
  const billingCycleEl = document.getElementById('billingCycle');
  const billingCycleContainer = document.getElementById('billingCycleContainer');
  const billingSeparator = document.getElementById('billingSeparator');

  if (currentPlan === 'growth') {
    // For Growth plan users
    const subscription = subscriptionState.subscription;

    // Show billing cycle container for Growth plan
    if (billingCycleContainer) billingCycleContainer.style.display = 'block';
    if (billingSeparator) billingSeparator.style.display = 'block';

    // Update billing period
    if (subscription?.current_period_start && subscription?.current_period_end) {
      const startDate = new Date(subscription.current_period_start).toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
      const endDate = new Date(subscription.current_period_end).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
      billingPeriodEl.textContent = `${startDate} - ${endDate}`;
    } else if (subscription?.current_period_end) {
      billingPeriodEl.textContent =
        `${new Date(subscription.current_period_end).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' })}`;
    } else {
      billingPeriodEl.textContent = '-';
    }

    // Update billing cycle
    if (subscription?.billing_cycle) {
      billingCycleEl.textContent = subscription.billing_cycle === 'annual' ? 'Annually' : 'Monthly';
    } else {
      billingCycleEl.textContent = isAnnual ? 'Annually' : 'Monthly';
    }
  } else {
    // For Starter (free) plan
    billingPeriodEl.textContent = 'Free Plan';

    // Hide billing cycle for free plan
    if (billingCycleContainer) billingCycleContainer.style.display = 'none';
    if (billingSeparator) billingSeparator.style.display = 'none';
  }

  // Update modal pricing
  document.getElementById('modalCurrency').textContent = currency.symbol;
  document.getElementById('modalAmount').textContent = formatNumber(displayPrice);
  document.getElementById('modalPeriod').textContent = isAnnual ? '/month (billed annually)' : '/month';
}

function updateUsageStats() {
  const usage = subscriptionState.usage;
  const plan = SUBSCRIPTION_CONFIG.PLANS[subscriptionState.currentPlan];
  const limits = plan.limits;

  // Orders
  const ordersLimit = limits.orders === -1 ? '∞' : limits.orders;
  document.getElementById('ordersUsed').textContent = usage.orders;
  document.getElementById('ordersLimit').textContent = ordersLimit;
  updateUsageBar('ordersBar', usage.orders, limits.orders);

  // Drivers
  const driversLimit = limits.drivers === -1 ? '∞' : limits.drivers;
  document.getElementById('driversUsed').textContent = usage.drivers;
  document.getElementById('driversLimit').textContent = driversLimit;
  updateUsageBar('driversBar', usage.drivers, limits.drivers);

  // Messages
  const messagesLimit = limits.whatsappMessages === -1 ? '∞' : limits.whatsappMessages;
  document.getElementById('messagesUsed').textContent = usage.whatsappMessages;
  document.getElementById('messagesLimit').textContent = messagesLimit;
  updateUsageBar('messagesBar', usage.whatsappMessages, limits.whatsappMessages);
}

function updateUsageBar(barId, used, limit) {
  const bar = document.getElementById(barId);
  if (limit === -1) {
    bar.style.width = '30%';
    bar.className = 'usage-bar-fill low';
    return;
  }

  const percentage = Math.min((used / limit) * 100, 100);
  bar.style.width = percentage + '%';

  if (percentage >= 90) {
    bar.className = 'usage-bar-fill high';
  } else if (percentage >= 70) {
    bar.className = 'usage-bar-fill medium';
  } else {
    bar.className = 'usage-bar-fill low';
  }
}

function updateBillingHistory() {
  const container = document.getElementById('billingHistoryBody');
  const history = subscriptionState.billingHistory;

  if (!history || history.length === 0) {
    container.innerHTML = '<div class="text-center p-lg text-secondary">No billing history yet</div>';
    return;
  }

  container.innerHTML = history.map(invoice => `
    <div class="invoice-row">
      <div>
        <div style="font-weight: 500;">${invoice.description}</div>
        <div class="invoice-date">${new Date(invoice.date).toLocaleDateString()}</div>
      </div>
      <div style="display: flex; align-items: center; gap: var(--space-md);">
        <span class="invoice-amount">${invoice.currency}${formatNumber(invoice.amount)}</span>
        <span class="invoice-status paid">Paid</span>
        <a href="${invoice.receipt_url}" target="_blank" class="btn btn-sm btn-secondary">Receipt</a>
      </div>
    </div>
  `).join('');
}

function checkLimits() {
  const usage = subscriptionState.usage;
  const plan = SUBSCRIPTION_CONFIG.PLANS[subscriptionState.currentPlan];
  const limits = plan.limits;

  let showWarning = false;
  let warningText = '';

  if (limits.orders !== -1 && usage.orders >= limits.orders * 0.8) {
    showWarning = true;
    warningText = `You've used ${usage.orders} of ${limits.orders} orders this month.`;
  } else if (limits.drivers !== -1 && usage.drivers >= limits.drivers) {
    showWarning = true;
    warningText = `You've reached your driver limit (${limits.drivers}).`;
  } else if (limits.whatsappMessages !== -1 && usage.whatsappMessages >= limits.whatsappMessages * 0.8) {
    showWarning = true;
    warningText = `You've used ${usage.whatsappMessages} of ${limits.whatsappMessages} WhatsApp messages.`;
  }

  const warningEl = document.getElementById('limitWarning');
  if (showWarning && subscriptionState.currentPlan === 'starter') {
    warningEl.style.display = 'flex';
    document.getElementById('limitWarningText').textContent = warningText + ' Upgrade to Growth for unlimited access.';
  } else {
    warningEl.style.display = 'none';
  }
}

// =============================================================================
// ACTIONS
// =============================================================================

function toggleBilling() {
  subscriptionState.isAnnual = !subscriptionState.isAnnual;

  const toggle = document.getElementById('billingToggle');
  const monthlyLabel = document.getElementById('monthlyLabel');
  const annualLabel = document.getElementById('annualLabel');

  toggle.classList.toggle('active', subscriptionState.isAnnual);
  monthlyLabel.classList.toggle('active', !subscriptionState.isAnnual);
  annualLabel.classList.toggle('active', subscriptionState.isAnnual);

  updatePlanCards();
}

function selectPlan(planId) {
  console.log('selectPlan called with:', planId);
  console.log('Current plan:', subscriptionState.currentPlan);

  if (planId === 'growth' && subscriptionState.currentPlan !== 'growth') {
    console.log('Opening upgrade modal...');
    showUpgradeModal();
  } else if (planId === 'starter' && subscriptionState.currentPlan === 'growth') {
    if (confirm('Are you sure you want to downgrade to Starter? You will lose access to premium features at the end of your billing period.')) {
      downgradeToStarter();
    }
  } else {
    console.log('No action needed - already on this plan or invalid state');
  }
}

function showUpgradeModal() {
  console.log('showUpgradeModal called');
  const modal = document.getElementById('upgradeModal');
  console.log('Modal element:', modal);
  modal.style.display = 'flex';
  modal.classList.add('active');
  console.log('Modal display set to flex and active class added');
}

function hideUpgradeModal() {
  const modal = document.getElementById('upgradeModal');
  modal.classList.remove('active');
  setTimeout(() => {
    modal.style.display = 'none';
  }, 300); // Wait for fade-out animation
}

function scrollToPlans() {
  document.getElementById('planGrid').scrollIntoView({ behavior: 'smooth' });
}

// Make functions globally available
window.selectPlan = selectPlan;
window.showUpgradeModal = showUpgradeModal;
window.hideUpgradeModal = hideUpgradeModal;
window.toggleBilling = toggleBilling;
window.scrollToPlans = scrollToPlans;

// =============================================================================
// PAYMENT PROCESSING (PAYSTACK)
// =============================================================================

async function processPayment() {
  const currency = subscriptionState.currency;
  const isAnnual = subscriptionState.isAnnual;
  const monthlyPrice = currency.monthlyPrice;
  const amount = isAnnual ? Math.round(monthlyPrice * 12 * 0.8) : monthlyPrice;
  
  // Get user email
  const user = JSON.parse(localStorage.getItem('numus_user') || '{}');
  const email = user.email;
  
  if (!email) {
    Utils.showToast('Please update your email in settings first', 'error');
    return;
  }

  // Paystack amount is in smallest currency unit (kobo for NGN, cents for USD, etc.)
  const amountInSmallestUnit = getAmountInSmallestUnit(amount, currency.code);

  const payButton = document.getElementById('payButton');
  const payButtonText = document.getElementById('payButtonText');
  payButtonText.textContent = 'Processing...';
  payButton.disabled = true;

  try {
    // Initialize Paystack
    const handler = PaystackPop.setup({
      key: SUBSCRIPTION_CONFIG.PAYSTACK_PUBLIC_KEY,
      email: email,
      amount: amountInSmallestUnit,
      currency: currency.code,
      ref: 'NUMUS_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9),
      metadata: {
        plan: 'growth',
        billing_cycle: isAnnual ? 'annual' : 'monthly',
        custom_fields: [
          {
            display_name: 'Plan',
            variable_name: 'plan',
            value: 'Growth'
          },
          {
            display_name: 'Billing Cycle',
            variable_name: 'billing_cycle',
            value: isAnnual ? 'Annual' : 'Monthly'
          }
        ]
      },
      callback: function(response) {
        handlePaymentSuccess(response);
      },
      onClose: function() {
        payButtonText.textContent = 'Pay Now';
        payButton.disabled = false;
      }
    });
    
    handler.openIframe();
  } catch (error) {
    console.error('Payment error:', error);
    Utils.showToast('Payment initialization failed. Please try again.', 'error');
    payButtonText.textContent = 'Pay Now';
    payButton.disabled = false;
  }
}

async function handlePaymentSuccess(response) {
  try {
    // Verify payment on backend
    const result = await api.verifySubscriptionPayment({
      reference: response.reference,
      plan: 'growth',
      billing_cycle: subscriptionState.isAnnual ? 'annual' : 'monthly'
    });

    if (result.success) {
      subscriptionState.currentPlan = 'growth';
      subscriptionState.subscription = result.data.subscription;
      
      hideUpgradeModal();
      updateUI();
      
      Utils.showToast('🎉 Welcome to Growth! Your upgrade is complete.', 'success');
      
      // Reload page to update all features
      setTimeout(() => location.reload(), 2000);
    } else {
      Utils.showToast('Payment verification failed. Please contact support.', 'error');
    }
  } catch (error) {
    Utils.showToast('Error processing payment. Please contact support.', 'error');
  }
  
  document.getElementById('payButtonText').textContent = 'Pay Now';
  document.getElementById('payButton').disabled = false;
}

async function downgradeToStarter() {
  try {
    const response = await api.cancelSubscription();
    if (response.success) {
      Utils.showToast('Your subscription will end at the current billing period.', 'success');
      await loadSubscriptionData();
      updateUI();
    }
  } catch (error) {
    Utils.showToast('Failed to cancel subscription. Please contact support.', 'error');
  }
}

// =============================================================================
// UTILITY FUNCTIONS
// =============================================================================

function formatNumber(num) {
  if (num >= 1000000) {
    return (num / 1000000).toFixed(1) + 'M';
  }
  if (num >= 1000) {
    return num.toLocaleString();
  }
  return num.toString();
}

function getAmountInSmallestUnit(amount, currencyCode) {
  // Most currencies use 100 subunits (cents, kobo, etc.)
  // Some currencies like JPY, KRW have no subunits
  const noSubunitCurrencies = ['JPY', 'KRW', 'VND', 'IDR'];
  
  if (noSubunitCurrencies.includes(currencyCode)) {
    return Math.round(amount);
  }
  
  return Math.round(amount * 100);
}

// =============================================================================
// LIMIT CHECKS (Export for use in other pages)
// =============================================================================

window.SubscriptionLimits = {
  async checkCanCreateOrder() {
    const plan = SUBSCRIPTION_CONFIG.PLANS[subscriptionState.currentPlan];
    const limits = plan.limits;
    
    if (limits.orders === -1) return { allowed: true };
    
    const usage = subscriptionState.usage;
    if (usage.orders >= limits.orders) {
      return {
        allowed: false,
        message: `You've reached your monthly order limit (${limits.orders}). Upgrade to Growth for unlimited orders.`,
        upgradeRequired: true
      };
    }
    
    return { allowed: true };
  },

  async checkCanAddDriver() {
    const plan = SUBSCRIPTION_CONFIG.PLANS[subscriptionState.currentPlan];
    const limits = plan.limits;
    
    if (limits.drivers === -1) return { allowed: true };
    
    const usage = subscriptionState.usage;
    if (usage.drivers >= limits.drivers) {
      return {
        allowed: false,
        message: `You've reached your driver limit (${limits.drivers}). Upgrade to Growth for unlimited drivers.`,
        upgradeRequired: true
      };
    }
    
    return { allowed: true };
  },

  checkFeatureAccess(feature) {
    const plan = SUBSCRIPTION_CONFIG.PLANS[subscriptionState.currentPlan];
    const limits = plan.limits;
    
    const featureMap = {
      'custom_domain': limits.customDomain,
      'payment_gateway': limits.paymentGateway,
      'cod_collection': limits.codCollection,
      'priority_support': limits.prioritySupport,
      'white_label': limits.whiteLabel,
      'api_access': limits.apiAccess
    };
    
    return featureMap[feature] || false;
  },

  getCurrentPlan() {
    return subscriptionState.currentPlan;
  },

  isGrowthPlan() {
    return subscriptionState.currentPlan === 'growth';
  }
};
