// Numus Logistics - Utility Functions

const Utils = {
  // Format currency
  formatCurrency(amount) {
    return new Intl.NumberFormat(CONFIG.CURRENCY.locale, {
      style: 'currency',
      currency: CONFIG.CURRENCY.code,
      minimumFractionDigits: 0,
      maximumFractionDigits: 2,
    }).format(amount);
  },

  // Format date
  formatDate(dateString, options = {}) {
    const date = new Date(dateString);
    const defaultOptions = {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    };
    return date.toLocaleDateString('en-NG', { ...defaultOptions, ...options });
  },

  // Format datetime
  formatDateTime(dateString) {
    const date = new Date(dateString);
    return date.toLocaleString('en-NG', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  },

  // Format time
  formatTime(dateString) {
    const date = new Date(dateString);
    return date.toLocaleTimeString('en-NG', {
      hour: '2-digit',
      minute: '2-digit',
    });
  },

  // Get relative time (e.g., "5 minutes ago")
  getRelativeTime(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now - date;
    const diffSecs = Math.floor(diffMs / 1000);
    const diffMins = Math.floor(diffSecs / 60);
    const diffHours = Math.floor(diffMins / 60);
    const diffDays = Math.floor(diffHours / 24);

    if (diffSecs < 60) return 'Just now';
    if (diffMins < 60) return `${diffMins} min${diffMins > 1 ? 's' : ''} ago`;
    if (diffHours < 24) return `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
    if (diffDays < 7) return `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;
    return this.formatDate(dateString);
  },

  // Get status badge HTML
  getStatusBadge(status, type = 'order') {
    const statuses = type === 'order' ? CONFIG.ORDER_STATUSES : CONFIG.DRIVER_STATUSES;
    const statusInfo = statuses[status] || { label: status, color: '#6b7280', bgColor: '#f3f4f6' };
    return `<span class="status-badge" style="color: ${statusInfo.color}; background-color: ${statusInfo.bgColor}">${statusInfo.label}</span>`;
  },

  // Get vehicle icon
  getVehicleIcon(vehicleType) {
    const vehicle = CONFIG.VEHICLE_TYPES[vehicleType];
    return vehicle ? vehicle.icon : '🚗';
  },

  // Truncate text
  truncate(text, maxLength = 50) {
    if (!text || text.length <= maxLength) return text;
    return text.substring(0, maxLength) + '...';
  },

  // Capitalize first letter
  capitalize(str) {
    if (!str) return '';
    return str.charAt(0).toUpperCase() + str.slice(1).replace(/_/g, ' ');
  },

  // Validate email
  isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  },

  // Validate phone (Nigerian format)
  isValidPhone(phone) {
    const phoneRegex = /^(\+234|0)[789][01]\d{8}$/;
    return phoneRegex.test(phone.replace(/\s/g, ''));
  },

  // Format phone number
  formatPhone(phone) {
    if (!phone) return '';
    const cleaned = phone.replace(/\D/g, '');
    if (cleaned.startsWith('234')) {
      return '+' + cleaned;
    }
    if (cleaned.startsWith('0')) {
      return '+234' + cleaned.substring(1);
    }
    return phone;
  },

  // Generate random ID
  generateId(prefix = '') {
    const timestamp = Date.now().toString(36);
    const random = Math.random().toString(36).substring(2, 8);
    return prefix ? `${prefix}-${timestamp}${random}` : `${timestamp}${random}`;
  },

  // Debounce function
  debounce(func, wait = 300) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  },

  // Show toast notification
  showToast(message, type = 'info', duration = 3000) {
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.innerHTML = `
      <div class="toast-content">
        <span class="toast-icon">${this.getToastIcon(type)}</span>
        <span class="toast-message">${message}</span>
      </div>
    `;
    
    document.body.appendChild(toast);
    
    // Trigger animation
    setTimeout(() => toast.classList.add('show'), 10);
    
    // Remove after duration
    setTimeout(() => {
      toast.classList.remove('show');
      setTimeout(() => toast.remove(), 300);
    }, duration);
  },

  getToastIcon(type) {
    const icons = {
      success: '✓',
      error: '✕',
      warning: '⚠',
      info: 'ℹ',
    };
    return icons[type] || icons.info;
  },

  // Show loading spinner
  showLoading(container, message = 'Loading...') {
    const loader = document.createElement('div');
    loader.className = 'loading-overlay';
    loader.innerHTML = `
      <div class="loading-spinner">
        <div class="spinner"></div>
        <p>${message}</p>
      </div>
    `;
    container.style.position = 'relative';
    container.appendChild(loader);
    return loader;
  },

  // Hide loading spinner
  hideLoading(loader) {
    if (loader) {
      loader.remove();
    }
  },

  // Confirm dialog
  async confirm(message, title = 'Confirm') {
    return new Promise((resolve) => {
      const modal = document.createElement('div');
      modal.className = 'modal-overlay active';
      modal.innerHTML = `
        <div class="modal confirm-modal">
          <div class="modal-header">
            <h3>${title}</h3>
          </div>
          <div class="modal-body">
            <p>${message}</p>
          </div>
          <div class="modal-footer">
            <button class="btn btn-secondary" data-action="cancel">Cancel</button>
            <button class="btn btn-primary" data-action="confirm">Confirm</button>
          </div>
        </div>
      `;
      
      document.body.appendChild(modal);
      
      modal.querySelector('[data-action="cancel"]').onclick = () => {
        modal.remove();
        resolve(false);
      };
      
      modal.querySelector('[data-action="confirm"]').onclick = () => {
        modal.remove();
        resolve(true);
      };
    });
  },

  // Parse query string
  parseQueryString(queryString = window.location.search) {
    const params = new URLSearchParams(queryString);
    const result = {};
    for (const [key, value] of params) {
      result[key] = value;
    }
    return result;
  },

  // Build query string
  buildQueryString(params) {
    return new URLSearchParams(params).toString();
  },

  // Calculate distance between two coordinates (Haversine formula)
  calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371; // Earth's radius in km
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) * Math.cos(this.toRad(lat2)) *
      Math.sin(dLon / 2) * Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return (R * c).toFixed(2);
  },

  toRad(deg) {
    return deg * (Math.PI / 180);
  },

  // Export data to CSV
  exportToCSV(data, filename = 'export.csv') {
    if (!data || !data.length) return;
    
    const headers = Object.keys(data[0]);
    const csvContent = [
      headers.join(','),
      ...data.map(row => 
        headers.map(header => {
          let cell = row[header] ?? '';
          // Escape quotes and wrap in quotes if contains comma
          if (typeof cell === 'string' && (cell.includes(',') || cell.includes('"'))) {
            cell = '"' + cell.replace(/"/g, '""') + '"';
          }
          return cell;
        }).join(',')
      )
    ].join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = filename;
    link.click();
    URL.revokeObjectURL(link.href);
  },

  // Format form data to object
  formDataToObject(form) {
    const formData = new FormData(form);
    const data = {};
    for (const [key, value] of formData) {
      // Handle numeric fields
      if (form.querySelector(`[name="${key}"]`)?.type === 'number') {
        data[key] = value ? parseFloat(value) : null;
      } else {
        data[key] = value;
      }
    }
    return data;
  },

  // Sanitize HTML to prevent XSS
  sanitizeHTML(str) {
    const temp = document.createElement('div');
    temp.textContent = str;
    return temp.innerHTML;
  },

  // Check if user is authenticated
  checkAuth() {
    if (!api.isAuthenticated()) {
      window.location.href = 'login';
      return false;
    }
    return true;
  },

  // Get current business info
  getCurrentBusiness() {
    return JSON.parse(localStorage.getItem('numus_business') || 'null');
  },

  // Get current user info
  getCurrentUser() {
    return JSON.parse(localStorage.getItem('numus_user') || 'null');
  },
};
